'use strict';

/* --------------------------------------------------------------
 filter.js 2016-09-01
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Handles the orders table filtering.
 *
 * ### Methods
 *
 * **Reload Filtering Options**
 *
 * ```
 * // Reload the filter options with an AJAX request (optionally provide a second parameter for the AJAX URL).
 * $('.table-main').orders_overview_filter('reload');
 * ```
 */
gx.controllers.module('filter', [jse.source + '/vendor/jquery-deparam/jquery-deparam.min.js'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Enter Key Code
     *
     * @type {Number}
     */

    var ENTER_KEY_CODE = 13;

    /**
     * Module Selector
     *
     * @type {jQuery}
     */
    var $this = $(this);

    /**
     * Filter Row Selector
     *
     * @type {jQuery}
     */
    var $filter = $this.find('tr.filter');

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = { bindings: {} };

    // Dynamically define the filter row data-bindings.
    $filter.find('th').each(function () {
        var columnName = $(this).data('columnName');

        if (columnName === 'checkbox' || columnName === 'actions') {
            return true;
        }

        module.bindings[columnName] = $(this).find('input, select').first();
    });

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Reload filter options with an Ajax request.
     *
     * This function implements the $('.datatable').orders_overview_filter('reload') which will reload the filtering
     * "multi_select" instances will new options. It must be used after some table data are changed and the filtering
     * options need to be updated.
     *
     * @param {String} url Optional, the URL to be used for fetching the options. Do not add the "pageToken"
     * parameter to URL, it will be appended in this method.
     */
    function _reload(url) {
        url = url || jse.core.config.get('appUrl') + '/admin/admin.php?do=OrdersOverviewAjax/FilterOptions';
        var data = { pageToken: jse.core.config.get('pageToken') };

        $.getJSON(url, data).done(function (response) {
            for (var column in response) {
                var $select = $filter.find('.SumoSelect > select.' + column);
                var currentValueBackup = $select.val(); // Will try to set it back if it still exists.

                if (!$select.length) {
                    return; // The select element was not found.
                }

                $select.empty();

                var _iteratorNormalCompletion = true;
                var _didIteratorError = false;
                var _iteratorError = undefined;

                try {
                    for (var _iterator = response[column][Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                        var option = _step.value;

                        $select.append(new Option(option.text, option.value));
                    }
                } catch (err) {
                    _didIteratorError = true;
                    _iteratorError = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion && _iterator.return) {
                            _iterator.return();
                        }
                    } finally {
                        if (_didIteratorError) {
                            throw _iteratorError;
                        }
                    }
                }

                if (currentValueBackup !== null) {
                    $select.val(currentValueBackup);
                }

                $select.multi_select('refresh');
            }
        });
    }

    /**
     * Add public "orders_overview_filter" method to jQuery in order.
     */
    function _addPublicMethod() {
        if ($.fn.orders_overview_filter) {
            return;
        }

        $.fn.extend({
            orders_overview_filter: function orders_overview_filter(action) {
                for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
                    args[_key - 1] = arguments[_key];
                }

                $.each(this, function () {
                    switch (action) {
                        case 'reload':
                            _reload.apply(this, args);
                            break;
                    }
                });
            }
        });
    }

    /**
     * On Filter Button Click
     *
     * Apply the provided filters and update the table rows.
     */
    function _onApplyFiltersClick() {
        // Prepare the object with the final filtering data.
        var filter = {};

        $filter.find('th').each(function () {
            var columnName = $(this).data('columnName');

            if (columnName === 'checkbox' || columnName === 'actions') {
                return true;
            }

            var value = module.bindings[columnName].get();

            if (value) {
                filter[columnName] = value;
                $this.DataTable().column(columnName + ':name').search(value);
            } else {
                $this.DataTable().column(columnName + ':name').search('');
            }
        });

        $this.trigger('orders_overview_filter:change', [filter]);
        $this.DataTable().draw();
    }

    /**
     * On Reset Button Click
     *
     * Reset the filter form and reload the table data without filtering.
     */
    function _onResetFiltersClick() {
        // Remove values from the input boxes.
        $filter.find('input, select').not('.length').val('');
        $filter.find('select').not('.length').multi_select('refresh');

        // Reset the filtering values.
        $this.DataTable().columns().search('').draw();

        // Trigger Event
        $this.trigger('orders_overview_filter:change', [{}]);
    }

    /**
     * Apply the filters when the user presses the Enter key.
     *
     * @param {jQuery.Event} event
     */
    function _onInputTextKeyUp(event) {
        if (event.which === ENTER_KEY_CODE) {
            $filter.find('.apply-filters').trigger('click');
        }
    }

    /**
     * Parse the initial filtering parameters and apply them to the table.
     */
    function _parseFilteringParameters() {
        var _$$deparam = $.deparam(window.location.search.slice(1)),
            filter = _$$deparam.filter;

        for (var name in filter) {
            var value = decodeURIComponent(filter[name]);

            if (module.bindings[name]) {
                module.bindings[name].set(value);
            }
        }
    }

    /**
     * Normalize array filtering values.
     *
     * By default datatables will concatenate array search values into a string separated with "," commas. This
     * is not acceptable though because some filtering elements may contain values with comma and thus the array
     * cannot be parsed from backend. This method will reset those cases back to arrays for a clearer transaction
     * with the backend.
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {DataTables.Settings} settings DataTables settings object.
     * @param {Object} data Data that will be sent to the server in an object form.
     */
    function _normalizeArrayValues(event, settings, data) {
        var filter = {};

        for (var name in module.bindings) {
            var value = module.bindings[name].get();

            if (value && value.constructor === Array) {
                filter[name] = value;
            }
        }

        for (var entry in filter) {
            var _iteratorNormalCompletion2 = true;
            var _didIteratorError2 = false;
            var _iteratorError2 = undefined;

            try {
                for (var _iterator2 = data.columns[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                    var column = _step2.value;

                    if (entry === column.name && filter[entry].constructor === Array) {
                        column.search.value = filter[entry];
                        break;
                    }
                }
            } catch (err) {
                _didIteratorError2 = true;
                _iteratorError2 = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion2 && _iterator2.return) {
                        _iterator2.return();
                    }
                } finally {
                    if (_didIteratorError2) {
                        throw _iteratorError2;
                    }
                }
            }
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Add public module method.
        _addPublicMethod();

        // Parse filtering GET parameters.
        _parseFilteringParameters();

        // Bind event handlers.
        $filter.on('keyup', 'input:text', _onInputTextKeyUp).on('click', '.apply-filters', _onApplyFiltersClick).on('click', '.reset-filters', _onResetFiltersClick);

        $this.on('preXhr.dt', _normalizeArrayValues);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
